<?php

class Maintenance {
    private $conn;
    private $table_name = "maintenance_history";

    public $id_maintenance;
    public $id_aset;
    public $tanggal_maintenance;
    public $tindakan;
    public $catatan_teknisi;
    public $nama_teknisi;
    public $created_at;

    // New properties for assessment details
    public $asset_category_for_assessment;
    public $assessment_data; // Will be a JSON string

    public function __construct($db) {
        $this->conn = $db;
    }

    // Create a new maintenance record
    public function create() {
        // Insert into maintenance_history table
        $query_history = "INSERT INTO " . $this->table_name . "
                          SET
                            id_aset=:id_aset,
                            tanggal_maintenance=:tanggal_maintenance,
                            tindakan=:tindakan,
                            catatan_teknisi=:catatan_teknisi,
                            nama_teknisi=:nama_teknisi";

        $stmt_history = $this->conn->prepare($query_history);

        // Sanitize
        $this->id_aset = htmlspecialchars(strip_tags($this->id_aset));
        $this->tanggal_maintenance = htmlspecialchars(strip_tags($this->tanggal_maintenance));
        $this->tindakan = htmlspecialchars(strip_tags($this->tindakan));
        $this->catatan_teknisi = htmlspecialchars(strip_tags($this->catatan_teknisi));
        $this->nama_teknisi = htmlspecialchars(strip_tags($this->nama_teknisi));

        // Bind values
        $stmt_history->bindParam(":id_aset", $this->id_aset);
        $stmt_history->bindParam(":tanggal_maintenance", $this->tanggal_maintenance);
        $stmt_history->bindParam(":tindakan", $this->tindakan);
        $stmt_history->bindParam(":catatan_teknisi", $this->catatan_teknisi);
        $stmt_history->bindParam(":nama_teknisi", $this->nama_teknisi);

        if ($stmt_history->execute()) {
            $this->id_maintenance = $this->conn->lastInsertId(); // Get the last inserted ID

            // If assessment data is provided, insert into maintenance_assessments table
            if (!empty($this->assessment_data) && !empty($this->asset_category_for_assessment)) {
                $query_assessment = "INSERT INTO maintenance_assessments
                                     SET
                                       id_maintenance=:id_maintenance,
                                       asset_category=:asset_category,
                                       assessment_data=:assessment_data";
                
                $stmt_assessment = $this->conn->prepare($query_assessment);

                // Sanitize assessment data
                $this->asset_category_for_assessment = htmlspecialchars(strip_tags($this->asset_category_for_assessment));
                // assessment_data is already expected to be JSON, no strip_tags

                $stmt_assessment->bindParam(":id_maintenance", $this->id_maintenance);
                $stmt_assessment->bindParam(":asset_category", $this->asset_category_for_assessment);
                $stmt_assessment->bindParam(":assessment_data", $this->assessment_data);

                if ($stmt_assessment->execute()) {
                    return true;
                } else {
                    // Optionally, you might want to log this failure or delete the maintenance_history record
                    // if assessment insertion is critical. For now, we return false.
                    return false;
                }
            }
            return true; // Return true if history inserted and no assessment data to insert
        }
        return false;
    }

    // Read all maintenance history for a specific asset, including assessment data
    public function readByAssetId($asset_id) {
        $query = "SELECT mh.*, ma.assessment_data, ma.asset_category
                  FROM " . $this->table_name . " mh
                  LEFT JOIN maintenance_assessments ma ON mh.id_maintenance = ma.id_maintenance
                  WHERE mh.id_aset = ?
                  ORDER BY mh.tanggal_maintenance DESC, mh.created_at DESC";

        $stmt = $this->conn->prepare($query);

        // Sanitize and bind
        $asset_id = htmlspecialchars(strip_tags($asset_id));
        $stmt->bindParam(1, $asset_id);

        $stmt->execute();
        return $stmt;
    }

    // Read all maintenance history for all assets, including asset details and assessment data
    public function readAllHistory() {
        $query = "SELECT mh.*, a.nama_perangkat, a.serial_number, a.kategori AS asset_kategori, ma.assessment_data
                  FROM " . $this->table_name . " mh
                  LEFT JOIN assets a ON mh.id_aset = a.id_aset
                  LEFT JOIN maintenance_assessments ma ON mh.id_maintenance = ma.id_maintenance
                  ORDER BY mh.tanggal_maintenance DESC, mh.created_at DESC";

        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt;
    }
}
